
import { useState } from 'react';
import { Search, Filter, Star, Clock, DollarSign, User } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Header } from '@/components/Header';
import { Footer } from '@/components/Footer';

const Gigs = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [selectedPriceRange, setSelectedPriceRange] = useState('all');

  const categories = [
    { value: 'all', label: 'All Categories' },
    { value: 'automation', label: 'AI Automation' },
    { value: 'integration', label: 'System Integration' },
    { value: 'analytics', label: 'Data Analytics' },
    { value: 'chatbots', label: 'Chatbots & AI' },
    { value: 'workflows', label: 'Workflow Design' }
  ];

  const priceRanges = [
    { value: 'all', label: 'All Prices' },
    { value: 'budget', label: '$50 - $500' },
    { value: 'standard', label: '$500 - $2000' },
    { value: 'premium', label: '$2000+' }
  ];

  const gigs = [
    {
      id: 1,
      title: "Custom AI Chatbot Development",
      description: "I'll create a sophisticated AI chatbot for your website that can handle customer inquiries, provide support, and generate leads 24/7.",
      provider: {
        name: "Alex Thompson",
        avatar: "photo-1519389950473-47ba0277781c",
        rating: 4.9,
        reviews: 127
      },
      image: "photo-1488590528505-98d2b5aba04b",
      price: 1200,
      deliveryTime: "7 days",
      category: "chatbots",
      tags: ["AI", "Chatbot", "Customer Service", "Lead Generation"],
      level: "Pro"
    },
    {
      id: 2,
      title: "Workflow Automation Setup",
      description: "Automate your repetitive business processes with custom workflow solutions that save time and reduce errors.",
      provider: {
        name: "Sarah Kim",
        avatar: "photo-1581091226825-a6a2a5aee158",
        rating: 4.8,
        reviews: 89
      },
      image: "photo-1461749280684-dccba630e2f6",
      price: 800,
      deliveryTime: "5 days",
      category: "workflows",
      tags: ["Automation", "Zapier", "Integration", "Efficiency"],
      level: "Expert"
    },
    {
      id: 3,
      title: "Data Analytics Dashboard",
      description: "Transform your raw data into actionable insights with custom analytics dashboards and reporting systems.",
      provider: {
        name: "Mike Chen",
        avatar: "photo-1498050108023-c5249f4df085",
        rating: 4.9,
        reviews: 156
      },
      image: "photo-1487058792275-0ad4aaf24ca7",
      price: 1500,
      deliveryTime: "10 days",
      category: "analytics",
      tags: ["Analytics", "Dashboard", "Data Visualization", "Reports"],
      level: "Expert"
    },
    {
      id: 4,
      title: "API Integration Service",
      description: "Connect your systems seamlessly with custom API integrations that streamline data flow between platforms.",
      provider: {
        name: "Emma Rodriguez",
        avatar: "photo-1605810230434-7631ac76ec81",
        rating: 4.7,
        reviews: 73
      },
      image: "photo-1518770660439-4636190af475",
      price: 600,
      deliveryTime: "4 days",
      category: "integration",
      tags: ["API", "Integration", "Automation", "Systems"],
      level: "Pro"
    },
    {
      id: 5,
      title: "Email Marketing Automation",
      description: "Set up intelligent email marketing campaigns that nurture leads and convert customers automatically.",
      provider: {
        name: "David Park",
        avatar: "photo-1526374965328-7f61d4dc18c5",
        rating: 4.8,
        reviews: 94
      },
      image: "photo-1488972685288-c3fd157d7c7a",
      price: 450,
      deliveryTime: "3 days",
      category: "automation",
      tags: ["Email Marketing", "Automation", "Lead Nurturing", "Campaigns"],
      level: "Standard"
    },
    {
      id: 6,
      title: "AI-Powered Content Generator",
      description: "Create an AI system that generates high-quality content for your blog, social media, and marketing materials.",
      provider: {
        name: "Lisa Wang",
        avatar: "photo-1487958449943-2429e8be8625",
        rating: 4.9,
        reviews: 112
      },
      image: "photo-1526374965328-7f61d4dc18c5",
      price: 950,
      deliveryTime: "6 days",
      category: "automation",
      tags: ["AI", "Content Generation", "Marketing", "Copywriting"],
      level: "Pro"
    }
  ];

  const filteredGigs = gigs.filter(gig => {
    const matchesSearch = gig.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         gig.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || gig.category === selectedCategory;
    const matchesPrice = selectedPriceRange === 'all' || 
                        (selectedPriceRange === 'budget' && gig.price >= 50 && gig.price <= 500) ||
                        (selectedPriceRange === 'standard' && gig.price > 500 && gig.price <= 2000) ||
                        (selectedPriceRange === 'premium' && gig.price > 2000);
    
    return matchesSearch && matchesCategory && matchesPrice;
  });

  return (
    <div className="min-h-screen bg-black text-white">
      <Header />
      
      {/* Hero Section */}
      <section className="pt-24 pb-8 px-4 bg-gradient-to-r from-purple-900/20 to-blue-900/20">
        <div className="container mx-auto text-center">
          <h1 className="text-4xl md:text-6xl font-bold mb-6">
            <span className="bg-gradient-to-r from-purple-400 to-blue-400 bg-clip-text text-transparent">
              AI Automation Gigs
            </span>
          </h1>
          <p className="text-xl text-gray-300 mb-8 max-w-3xl mx-auto">
            Discover expert freelancers ready to transform your business with cutting-edge AI automation solutions
          </p>
        </div>
      </section>

      {/* Search and Filters */}
      <section className="py-8 px-4 border-b border-gray-800">
        <div className="container mx-auto">
          <div className="flex flex-col lg:flex-row gap-4 items-center justify-between">
            <div className="relative flex-1 max-w-md">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
              <Input
                placeholder="Search for AI automation services..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 bg-gray-900 border-gray-700 text-white"
              />
            </div>
            
            <div className="flex gap-4">
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger className="w-48 bg-gray-900 border-gray-700 text-white">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent className="bg-gray-900 border-gray-700">
                  {categories.map(category => (
                    <SelectItem key={category.value} value={category.value} className="text-white">
                      {category.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select value={selectedPriceRange} onValueChange={setSelectedPriceRange}>
                <SelectTrigger className="w-40 bg-gray-900 border-gray-700 text-white">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent className="bg-gray-900 border-gray-700">
                  {priceRanges.map(range => (
                    <SelectItem key={range.value} value={range.value} className="text-white">
                      {range.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      </section>

      {/* Results */}
      <section className="py-8 px-4">
        <div className="container mx-auto">
          <div className="mb-6">
            <h2 className="text-2xl font-semibold text-white">
              {filteredGigs.length} Services Available
            </h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredGigs.map((gig) => (
              <Card key={gig.id} className="bg-gray-900 border-gray-800 hover:border-purple-500/50 transition-all duration-300 hover:transform hover:scale-105">
                <div className="relative">
                  <img 
                    src={`https://images.unsplash.com/${gig.image}?auto=format&fit=crop&w=400&q=80`}
                    alt={gig.title}
                    className="w-full h-48 object-cover rounded-t-lg"
                  />
                  <Badge className="absolute top-3 right-3 bg-purple-600 hover:bg-purple-700">
                    {gig.level}
                  </Badge>
                </div>
                
                <CardHeader className="pb-3">
                  <h3 className="text-lg font-semibold text-white line-clamp-2">
                    {gig.title}
                  </h3>
                  <p className="text-gray-400 text-sm line-clamp-2">
                    {gig.description}
                  </p>
                </CardHeader>

                <CardContent className="pt-0">
                  <div className="flex items-center gap-2 mb-3">
                    <img 
                      src={`https://images.unsplash.com/${gig.provider.avatar}?auto=format&fit=crop&w=32&q=80`}
                      alt={gig.provider.name}
                      className="w-6 h-6 rounded-full object-cover"
                    />
                    <span className="text-sm text-gray-300">{gig.provider.name}</span>
                    <div className="flex items-center gap-1 ml-auto">
                      <Star className="h-4 w-4 text-yellow-400 fill-current" />
                      <span className="text-sm text-gray-300">{gig.provider.rating}</span>
                      <span className="text-sm text-gray-500">({gig.provider.reviews})</span>
                    </div>
                  </div>

                  <div className="flex flex-wrap gap-1 mb-4">
                    {gig.tags.slice(0, 3).map((tag, index) => (
                      <Badge key={index} variant="secondary" className="text-xs bg-gray-800 text-gray-300">
                        {tag}
                      </Badge>
                    ))}
                  </div>

                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4 text-sm text-gray-400">
                      <div className="flex items-center gap-1">
                        <Clock className="h-4 w-4" />
                        {gig.deliveryTime}
                      </div>
                    </div>
                    <div className="text-right">
                      <div className="text-xl font-bold text-white">
                        ${gig.price}
                      </div>
                    </div>
                  </div>

                  <Button className="w-full mt-4 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700">
                    Order Now
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredGigs.length === 0 && (
            <div className="text-center py-12">
              <div className="text-gray-400 text-lg mb-4">No services found matching your criteria</div>
              <Button 
                onClick={() => {
                  setSearchTerm('');
                  setSelectedCategory('all');
                  setSelectedPriceRange('all');
                }}
                variant="outline"
                className="border-purple-500 text-purple-400 hover:bg-purple-500 hover:text-white"
              >
                Clear Filters
              </Button>
            </div>
          )}
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default Gigs;
